<?php
namespace App\Http\Controllers\Api\Auth;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Twilio\Rest\Client;
use App\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Str;
use Illuminate\Support\Carbon;
use App\PaypalSubscription;
use Mail;
use App\Mail\verifyEmail;

class RegisterController extends Controller
{
    use IssueTokenTrait;
    private $otpDir;
    private $config;
    private $client;
    public function __construct()
    {
        $this->config = Config::first();
        $this->otpDir = public_path('otp'); // Use public directory for OTP files
        $this->client = \Laravel\Passport\Client::find(2);
        if (!File::exists($this->otpDir)) {
            File::makeDirectory($this->otpDir, 0755, true);
        }
    }
    private function isTwilioEnabled()
    {
        return $this->config->otp_enable ?? false;
    }
    private function generateOTP()
    {
        return rand(100000, 999999);
    }
    private function storeOTPAndData($mobile, $otp, $data = null)
    {
        $filename = $this->otpDir . '/' . $mobile . '.txt';
        $content = $otp . "\n";
        if ($data && isset($data['mobile'])) {
            // Don't encrypt data if logging in with mobile number
            $content .= json_encode($data) . "\n"; // Store data without encryption
        } else {
            // Encrypt data if it's not a mobile login
            $encryptedData = Crypt::encrypt(json_encode($data));
            $content .= $encryptedData . "\n";
        }
        if ($data) {
            $encryptedData = Crypt::encrypt(json_encode($data));
            $content .= $encryptedData . "\n";
        }
        $expirationTime = time() + 60; // 1 minute from now
        $content .= $expirationTime;
        File::put($filename, $content);
        // Start a background process to delete the file after 1 minute
        $this->startBackgroundDeletion($filename, $expirationTime);
    }
    private function startBackgroundDeletion($filename, $expirationTime)
    {
        $command = "php -r \"sleep(" . ($expirationTime - time()) . "); " .
            "if (file_exists('$filename')) { unlink('$filename'); }\" > /dev/null 2>&1 & echo $!";
        exec($command);
    }
    private function getOTPAndData($mobile)
    {
        $filename = $this->otpDir . '/' . $mobile . '.txt';
        if (File::exists($filename)) {
            $content = File::get($filename);
            $lines = explode("\n", trim($content));
            $otp = $lines[0]; // First line is the OTP
            $data = null;
            $expirationTime = null;
            if (count($lines) > 1) {
                $data = json_decode(Crypt::decrypt($lines[1]), true); // Second line is encrypted data
            }
            if (count($lines) > 2) {
                $expirationTime = (int)$lines[2]; // Third line is expiration timestamp
            }
            // Check if OTP has expired
            if ($expirationTime && time() > $expirationTime) {
                return null; // OTP expired
            }
            return [
                'otp' => $otp,
                'data' => $data
            ];
        }
        return null; // File not found
    }
    private function deleteOTPFile($mobile)
    {
        $filename = $this->otpDir . '/' . $mobile . '.txt';
        if (File::exists($filename)) {
            File::delete($filename);
        }
    }
    public function register(Request $request)
    {
        $this->validate($request, [
            'name' => 'required',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|min:6',
        ]);
        $user = User::create([
            'name' => request('name'),
            'email' => request('email'),
            'password' => bcrypt(request('password')),
            'is_blocked' => '0',
            'status' => '1',
            'verifyToken' => Str::random(5),
        ]);
        $user->assignRole('User');
        $config = Config::first();
        if ($config->free_sub == 1) {
            $this->freesubscribe($user);
        }
        if ($config->verify_email == 1) {
            $thisuser = User::findOrfail($user->id);
            $thisuser->status = 0;
            $thisuser->save();
            try {
                Mail::to($user->email)->send(new verifyEmail($user));
                return response()->json('Verification email sent !', 301);
            } catch (\Exception $e) {
                //return $e->getMessage();
                return response()->json('Mail Sending Error', 400);
            }
            return $this->issueToken($request, 'password');
        }
        if ($config->wel_eml == 1) {
            try {
                Mail::to(request('email'))->send(new WelcomeUser($user));
                return $this->issueToken($request, 'password');
            } catch (\Exception $e) {
                return $this->issueToken($request, 'password');
                return response()->json('Registraion successfull but mail not sent !', 200);
            }
        }
        if ($config->wel_eml == 0 && $config->verify_email == 0) {
            return $this->issueToken($request, 'password');
        }
    }
    public function freesubscribe($thisuser)
    {
        $config = Config::first();
        $start = Carbon::now();
        $end = $start->addDays($config->free_days);
        $payment_id = mt_rand(10000000000000, 99999999999999);
        $created_subscription = PaypalSubscription::create([
            'user_id' => $thisuser->id,
            'payment_id' => $payment_id,
            'user_name' => $thisuser->name,
            'package_id' => 100,
            'price' => 0,
            'status' => 1,
            'method' => 'free',
            'subscription_from' => Carbon::now(),
            'subscription_to' => $end
        ]);
    }
    public function verifyemail(Request $request)
    {
        $user = User::where(['email' => $request->email, 'verifyToken' => $request->token])->first();
        if ($user) {
            $user->status = 1;
            $user->verifyToken = NULL;
            $user->save();
            Mail::to($user->email)->send(new WelcomeUser($user));
            return $this->issueToken($request, 'password');
        } else {
            return response()->json('user not found', 401);
        }
    }
    public function resendOTP(Request $request)
    {
        $mobile = $request->input('mobile');
        if (!$mobile) {
            return response()->json(['error' => 'No phone number provided for OTP resend.'], 400);
        }
        $storedData = $this->getOTPAndData($mobile);
        if (!$storedData) {
            return response()->json(['error' => 'No existing OTP data found.'], 400);
        }
        $newOtp = $this->generateOTP();
        try {
            $accountSid = env('TWILIO_SID');
            $authToken = env('TWILIO_AUTH_TOKEN');
            $twilioNumber = env('TWILIO_NUMBER');
            $client = new Client($accountSid, $authToken);
            $client->messages->create(
                $mobile,
                [
                    'from' => $twilioNumber,
                    'body' => "Your new OTP is: $newOtp"
                ]
            );
            $this->storeOTPAndData($mobile, $newOtp, $storedData['data']);
            return response()->json(['message' => 'New OTP sent.'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to resend OTP: ' . $e->getMessage()], 500);
        }
    }
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required',
            'password' => 'required'
        ]);
        $authUser = User::where('email', $request->email)->first();
        if (!$authUser) {
            return response()->json(['message' => 'Unauthorized!'], 401);
        }
        if ($authUser->is_blocked == 1) {
            return response()->json(['message' => 'Blocked User'], 403);
        }
        if ($authUser->status == 0) {
            return response()->json(['message' => 'Please verify your email!'], 403);
        }
        if (Hash::check($request->password, $authUser->password)) {
            return $this->issueToken($request, 'password');
        }
        return response()->json(['message' => 'Unauthorized!'], 401);
    }
    public function refresh(Request $request){
        $this->validate($request, [
            'refresh_token' => 'required'
        ]);

        return $this->issueToken($request, 'refresh_token');
    }
    public function initiateOTPLogin(Request $request)
    {
        if (!$this->isTwilioEnabled()) {  // Add this check
            return response()->json(['error' => 'OTP login is not enabled.'], 400);
        }
        $request->validate([
            'mobile' => 'required|string'
        ]);
        $mobile = $request->input('mobile');
        $otp = $this->generateOTP();
        try {
            $accountSid = env('TWILIO_SID');
            $authToken = env('TWILIO_AUTH_TOKEN');
            $twilioNumber = env('TWILIO_NUMBER');
            $client = new Client($accountSid, $authToken);
            $client->messages->create(
                $mobile,
                [
                    'from' => $twilioNumber,
                    'body' => "Your OTP for login is: $otp"
                ]
            );
            $this->storeOTPAndData($mobile, $otp); // No need to store user data
            return response()->json(['message' => 'OTP sent for login. Please verify your phone number.'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to send OTP: ' . $e->getMessage()], 500);
        }
    }
    public function verifyOTP(Request $request)
    {
        $request->validate([
            'otp' => 'required|numeric',
            'mobile' => 'required|string',
        ]);
        $storedData = $this->getOTPAndData($request->mobile);
        if (!$storedData) {
            return response()->json(['error' => 'OTP is expired or invalid.'], 400);
        }
        if ($storedData['otp'] == $request->otp) {
            $user = User::where('mobile', $request->mobile)->first();
            if ($user) {
                Auth::login($user);
                $token = $user->createToken('API Token')->accessToken;
                return response()->json(['message' => 'Login successful!', 'token' => $token], 200);
            }
            return response()->json(['error' => 'User not found.'], 404);
        }
        return response()->json(['error' => 'Invalid OTP.'], 400);
    }
}